<?php

namespace App\Http\Controllers;

use App\Models\Item;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ItemController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $items = Item::orderBy('id','desc')->paginate(3);
        return view('item/all', compact('items'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('item/new');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //validate
        $request->validate([
            'name' => 'required|string|max:100',
            'min_qty' => 'integer|gt:0',
            'max_qty' => 'integer|gt:0',
            'note' => 'string|nullable|max:200',
            'image_name' => 'image|mimes:jpg,jpeg,png|max:2048'
        ]);


        $item = [
            'name' => $request->name,
            'min_qty' => $request->min_qty,
            'max_qty' => $request->max_qty,
            'note' => $request->note,
        ];

        //if exist image upload then set and upload
        if ($request->hasFile('picture')) {
            //upload file and then save document name and note in database
            $imageName = $request->file('picture')->store('images', 'public');
            $item['image_name'] = $imageName;
        }

        $item = Item::create($item);
        $item->code =  "ITM-" . str_pad($item->id, 5, '0', STR_PAD_LEFT); // LN-001, LN-002 etc.
        $item->save();

        return redirect('item/all')->with('success', 'Item data created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Item  $item
     * @return \Illuminate\Http\Response
     */
    public function show(Item $item, $id)
    {
        $item = Item::findOrFail($id);
        return view('item/view', compact('item'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Item  $item
     * @return \Illuminate\Http\Response
     */
    public function edit(Item $item, $id)
    {
        $item = Item::findOrFail($id);
        return view('item/edit', compact('item'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Item  $item
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {

        //validate
        $request->validate([
            'name' => ['required', 'string', 'max:100', Rule::unique('items', 'name')->ignore($id)],
            'min_qty' => 'integer|gt:0',
            'max_qty' => 'integer|gt:0',
            'note' => 'string|nullable|max:200',
            'image_name' => 'image|mimes:jpg,jpeg,png|max:2048'
        ]);


        $item = Item::findOrFail($id);
        $item->name = $request->name;
        $item->min_qty = $request->min_qty;
        $item->max_qty = $request->max_qty;
        $item->note = $request->note;



        //if image is selected in file input
        if ($request->hasFile('picture')) {
            //check previouse image is exist then remove it from uploaded location first
            if ($item->image_name != null && Storage::disk('public')->exists($item->image_name)) {
                Storage::disk('public')->delete($item->image_name);
                //previouse image is exist  then remove previouse data first
            }

            $imageName = $request->file('picture')->store('images', 'public');
            $item['image_name'] = $imageName;
        } else {
            if ($request->has('picture_remove')) {
                if (($item->image_name != "" || $item->image_name != null) && Storage::disk('public')->exists($item->image_name)) {
                    $item->image_name = '';
                    Storage::disk('public')->delete($item->image_name);
                }
            }
        }


        $item->update();
        return redirect('item/all')->with('success', 'item data upated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Item  $item
     * @return \Illuminate\Http\Response
     */
    public function destroy(Item $item, $id)
    {
        //
        $item = Item::findOrFail($id);
        //check image is exist
        if ($item->image_name != null && Storage::disk('public')->exists($item->image_name)) {
            Storage::disk('public')->delete($item->image_name);
        }
        $item->delete();
        return redirect('item/all')->with('success', 'item data deleted successfully');
    }
}
